function vci = nhkRender(vci,isa)
% Color rendering pipeline to convert sensor data to display image
% This script is only modified in the sense that it calls our custom
% demosaicing routine, nhkDemosaic, which calls the appropriate script
% Hiroshi Shimamoto and Kunal Ghosh
%
%    vci = myRender(vci,isa)
%
%  This is an example of a script that you can customize for your own
%  processing pipeline. This routine applies demosaic, color space
%  conversion to an internal representation, color balancing and color
%  conversion to a display representation (in that order).  
%
%  If the sensor is monochrome, the pipeline copies the isa data to the display
%  output as a monochrome image.
%
%  We will be writing other rendering pipelines based on different
%  architectures in the future.
%
%  This method runs on the assumption that the internal color space is
%  three-dimensional.  
%
% Copyright ImagEval Consultants, LLC, 2003.

% TODO
% The processing chains repeatedly transforms the vci.data.input,
% over-writing itself.  Perhaps we should be making an array of images,
% keeping them all, and ultimately examining them separately?


disp('nhkRender')

% Normally, we demosaic the quantized values.  If this field is empty, we
% warn the user that we are using the continuous voltages and then carry on.
vci = imageSet(vci,'input',sensorGet(isa,'dvorvolts'));

%  This is either the max digital value or the voltage swing, depending on
%  whether we have computed DVs or Volts.
vci = imageSet(vci,'datamax',sensorGet(isa,'max'));

nFilters = sensorGet(isa,'nfilters');
if nFilters == 1
    % If monochrome sensor, just copy the sensor values to the RGB values
    % of the display
    [r,c] = size(vci.data.input);
    vci = imageSet(vci,'result',repmat(vci.data.input,[1,1,3]));
    return;
end

if nFilters == 2
    warndlg('Rendering pipeline not implemented for 2 color sensor data.');
    vci = imageSet(vci,'result',[]);
    return;
end

% Basic color processing pipeline.  We start with the information about
% algorithms in the vci and the information from the image sensor array.
% We compute the demosaicedImage first.  Then we convert it to an internal
% color space.  Then we color balance the image.  Then we convert it to the
% display space.  Each of these routines has various algorithms available,
% including possible customer-supplied algorithms that can be specified by
% the popup menus.
if nFilters >= 3
    
    %1.  Demosaic first.
    %    img was vci.data.input in the next routine.
    img = nhkDemosaic(vci,isa); % Call nhkDemosaic routine
    
    % 2.  Convert the data into the internal color space
    [img,vci] = imageColorConversion(img,vci,isa);

    % 3.  Color balance in the internal color space
    [img,vci] = imageColorBalance(img,vci,isa);

    % 4.  Convert from the target space into display space.  The target
    % space is a calibrated space, and the display properties are
    % specified.  So, this could be done precisely. N.B. The display on the
    % user's desk is not likely to be the calibrated display that is
    % specified.
    [img,vci] = displayRender(img,vci,isa);

end

img = ieClip(img,0,1);

vci = imageSet(vci,'result',img);

return;